DOUBLE PRECISION FUNCTION norma_vetor(array, n)
    !Calcula a norma de um dado vetor, com dada dimensao n

    IMPLICIT NONE

    !Declaração
    INTEGER, INTENT(IN) :: n
    DOUBLE PRECISION, DIMENSION(n), INTENT(IN) :: array

    !Cálculo da norma
    norma_vetor = SQRT(SUM(array * array))

END FUNCTION norma_vetor

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

SUBROUTINE eliminacao(matriz, n, array_b)
    !Recebe uma matriz de coeficientes nxn, a triangulariza por pivotamento parcial e realiza as mesmas operacoes no vetor array_b de termos independentes.
    
    IMPLICIT NONE
    
    DOUBLE PRECISION, DIMENSION(n,n) :: matriz !Matriz a ser triangularizada
    DOUBLE PRECISION, DIMENSION(n) :: aux_matriz, array_b !Array auxiliar nos cálculos e array de termos independentes
    DOUBLE PRECISION :: x, aux_array !Variáveis auxiliares de cálculos
    INTEGER :: n, i, j !Dimensão n da matriz e dos arrays acima; variáveis auxiliares dos cálculos
    
    !Laço que varre linhas da matriz
    DO i = 1, n

        !Laço que varre as linhas adjacentes
        j = i + 1    

        DO WHILE (j <= n)
        
            !Compara "candidatos" a pivô parcial
            IF (ABS(matriz(i,i)) < ABS(matriz(j,i))) THEN
            
                !Troca as linhas colocando na posição correta a qual contém o maior pivô e troca as mesmas linhas no array_b
                aux_matriz(i:n) = matriz(i,i:n)
                matriz(i,i:n) = matriz(j,i:n)
                matriz(j,i:n) = aux_matriz(i:n)
                
                aux_array = array_b(i)
                array_b(i) = array_b(j)
                array_b(j) = aux_array

            END IF

            j = j + 1

        END DO

        !Laço que faz a eliminação
        j = i + 1

        DO WHILE (j <= n)
        
            !Define o multiplicador x do pivô, faz a eliminação e aplica o multiplicador no array_b
            x = matriz(j,i) / matriz(i,i)
            matriz(j,i:n) = matriz(i,i:n) * x - matriz(j,i:n)
            array_b(j) = array_b(i) * x - array_b(j)
            
            j = j + 1

        END DO
    
    END DO

END SUBROUTINE eliminacao

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

SUBROUTINE substituicao(matriz, array_x, array_b, n)
    !Recebe uma matriz nxn e dois arrays de dimensão n. A partir da matriz e array_b, encontra a solução array_x de um sistema linear pelo método de Gauss (já feita a etapa de eliminação)

    IMPLICIT NONE

    DOUBLE PRECISION, DIMENSION(n,n) :: matriz !Matriz já triangularizada
    DOUBLE PRECISION, DIMENSION(n), INTENT(OUT) :: array_x !Array solução do sistema
    DOUBLE PRECISION, DIMENSION(n) :: array_b !Array b de termos independentes, modificado pela triangularizacao de A
    DOUBLE PRECISION :: soma, aux !Variáveis auxiliares dos cálculos
    INTEGER :: n, i, j !Dimensão n dos arrays/matriz acima e variáveis auxiliares

    !Determina os elementos do array_x por back substitution
    array_x(n) = array_b(n) / matriz(n,n)
    
    DO i = n - 1, 1, -1

        !Laço que determina a somatoria no cálculo dos elementos array_x(i)
        j = i + 1
        soma = 0.

        DO WHILE (j <= n)

            soma = soma + matriz(i,j) * array_x(j)

            j = j + 1

        END DO

        !Cálculo propriamente dito
        array_x(i) = (array_b(i) - soma) / matriz(i,i)

    END DO

END SUBROUTINE substituicao

!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

PROGRAM gauss
    !Resolvendo um sistema linear pelo método de eliminação de Gauss

    IMPLICIT NONE

    DOUBLE PRECISION, DIMENSION(4,4) :: matrizA, matrizA_norm !Matriz de coeficientes original e normalizada
    DOUBLE PRECISION, DIMENSION(4) :: vetorX, vetorB !Vetores de incógnitas (solução) e de termos independentes
    INTEGER :: i, j !Variáveis auxiliares
    DOUBLE PRECISION, EXTERNAL :: norma_vetor !Função que calcula a norma de um vetor
    DOUBLE PRECISION :: mod_det, x !Módulo do determinante da matriz normalizada e valor x usado para função EPSILON

    !Definindo o sistema
    matrizA = RESHAPE((/4., 2., 2., 6., 3., 1., 2., 1., 2., 1., 2., 1., 2., 2., 4., 4./) , (/4, 4/))
    vetorB = (/5., 8., 3., 1./)

    !CONDICIONAMENTO DO SISTEMA
    !Primeiro: normalização da matriz de coeficientes
    DO i = 1, 4
        
        matrizA_norm(i,:) = matrizA(i,:) / norma_vetor(matrizA(i,:), 4)
        vetorB(i) = vetorB(i) / norma_vetor(matrizA(i,:), 4)

    END DO

    !Segundo: encontrando o determinante da matriz normalizada
    CALL eliminacao(matrizA_norm, 4, vetorB)
    mod_det = 1.

    DO i = 1, 4

        mod_det = mod_det * matrizA_norm(i,i)

    END DO

    mod_det = ABS(mod_det)

    !Verificando condicionamento e, caso bem condicionado, resolve o sistema
    IF (mod_det > (SQRT(64.) * EPSILON(x))) THEN

        !CALL eliminacao(matrizA, 4, pivotamento)
        CALL substituicao(matrizA_norm, vetorX, vetorB, 4)

        PRINT *, "A solução encontrada para o sistema foi:"
        PRINT *, vetorX

    ELSE

        PRINT *, "O sistema é mal condicionado."

    END IF

END PROGRAM gauss
